import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Calendar, DollarSign, Bike, Star, MessageSquare, TrendingUp, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const [rentals, setRentals] = useState({ active: [], past: [] })
  const [activeTab, setActiveTab] = useState('active')

  useEffect(() => {
    loadRentals()
    const interval = setInterval(loadRentals, 500)
    return () => clearInterval(interval)
  }, [])

  const loadRentals = () => {
    const saved = localStorage.getItem('bikeRentals')
    if (saved) {
      try {
        setRentals(JSON.parse(saved))
      } catch (e) {
        setRentals({ active: [], past: [] })
      }
    } else {
      const defaultRentals = {
        active: [
          {
            id: 1,
            model: 'Trek Mountain Pro',
            type: 'Горный',
            date: '2025-03-28',
            returnDate: '2025-04-05',
            price: 1500,
            status: 'active'
          },
          {
            id: 2,
            model: 'City Cruiser 2024',
            type: 'Городской',
            date: '2025-03-25',
            returnDate: '2025-04-02',
            price: 1200,
            status: 'active'
          }
        ],
        past: [
          {
            id: 3,
            model: 'Electric Speed X1',
            type: 'Электровелосипед',
            date: '2025-03-15',
            returnDate: '2025-03-20',
            price: 2000,
            review: 'Отличный велосипед, очень удобный!'
          },
          {
            id: 4,
            model: 'Kids Bike Junior',
            type: 'Детский',
            date: '2025-03-10',
            returnDate: '2025-03-12',
            price: 800,
            review: 'Ребенок в восторге!'
          }
        ]
      }
      setRentals(defaultRentals)
      localStorage.setItem('bikeRentals', JSON.stringify(defaultRentals))
    }
  }

  const stats = [
    { 
      icon: Bike, 
      value: rentals.past.length, 
      label: 'Завершено аренд',
      color: '#22C55E'
    },
    { 
      icon: DollarSign, 
      value: rentals.past.reduce((sum, r) => sum + (r.price || 0), 0), 
      label: 'Потрачено',
      color: '#16A34A'
    },
    { 
      icon: Star, 
      value: rentals.past.filter(r => r.review).length, 
      label: 'Оставлено отзывов',
      color: '#4ADE80'
    }
  ]

  return (
    <div className="my-rentals">
      <div className="page-layout">
        <aside className="sidebar">
          <div className="sidebar-header">
            <h2 className="sidebar-title">Статистика</h2>
          </div>
          <div className="sidebar-stats">
            {stats.map((stat, index) => (
              <motion.div
                key={stat.label}
                className="sidebar-stat-card"
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ x: 5, boxShadow: '0 8px 24px rgba(34, 197, 94, 0.2)' }}
              >
                <div className="sidebar-stat-icon" style={{ background: `${stat.color}20` }}>
                  <stat.icon className="stat-icon" size={24} color={stat.color} />
                </div>
                <div className="sidebar-stat-content">
                  <div className="sidebar-stat-value">
                    {stat.label === 'Потрачено' ? `${stat.value} ₽` : stat.value}
                  </div>
                  <div className="sidebar-stat-label">{stat.label}</div>
                </div>
              </motion.div>
            ))}
          </div>
        </aside>

        <main className="main-content-area">
          <section className="page-header">
            <motion.div
              className="header-content"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
            >
              <h1 className="page-title">Мои аренды</h1>
              <p className="page-subtitle">
                Управляйте своими арендами велосипедов
              </p>
            </motion.div>
          </section>

          <div className="tabs-container">
            <button
              className={`tab-button ${activeTab === 'active' ? 'active' : ''}`}
              onClick={() => setActiveTab('active')}
            >
              <Bike size={18} />
              <span>Активные ({rentals.active.length})</span>
            </button>
            <button
              className={`tab-button ${activeTab === 'past' ? 'active' : ''}`}
              onClick={() => setActiveTab('past')}
            >
              <Calendar size={18} />
              <span>История ({rentals.past.length})</span>
            </button>
          </div>

          <div className="rentals-list">
            {activeTab === 'active' ? (
              rentals.active.length > 0 ? (
                rentals.active.map((rental) => (
                  <motion.div
                    key={rental.id}
                    className="rental-item"
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    whileHover={{ x: 5, boxShadow: '0 8px 24px rgba(34, 197, 94, 0.15)' }}
                  >
                    <div className="rental-item-left">
                      <div className="rental-item-icon">
                        <Bike size={32} color="#22C55E" />
                      </div>
                      <div className="rental-item-info">
                        <h3 className="rental-item-model">{rental.model}</h3>
                        <p className="rental-item-type">{rental.type}</p>
                      </div>
                    </div>
                    <div className="rental-item-center">
                      <div className="rental-item-date">
                        <Calendar size={16} color="#78716C" />
                        <span>{new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })} - {new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })}</span>
                      </div>
                    </div>
                    <div className="rental-item-right">
                      <div className="rental-item-price">{rental.price} ₽</div>
                      <span className="rental-item-status">Активна</span>
                    </div>
                  </motion.div>
                ))
              ) : (
                <div className="empty-state">
                  <Bike size={64} color="#D1D5DB" />
                  <p>У вас нет активных аренд</p>
                </div>
              )
            ) : (
              rentals.past.length > 0 ? (
                rentals.past.map((rental) => (
                  <motion.div
                    key={rental.id}
                    className="rental-item"
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    whileHover={{ x: 5, boxShadow: '0 8px 24px rgba(34, 197, 94, 0.15)' }}
                  >
                    <div className="rental-item-left">
                      <div className="rental-item-icon">
                        <Bike size={32} color="#22C55E" />
                      </div>
                      <div className="rental-item-info">
                        <h3 className="rental-item-model">{rental.model}</h3>
                        <p className="rental-item-type">{rental.type}</p>
                        {rental.review && (
                          <div className="rental-item-review">
                            <MessageSquare size={14} color="#22C55E" />
                            <span>{rental.review}</span>
                          </div>
                        )}
                      </div>
                    </div>
                    <div className="rental-item-center">
                      <div className="rental-item-date">
                        <Calendar size={16} color="#78716C" />
                        <span>{new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })} - {new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })}</span>
                      </div>
                    </div>
                    <div className="rental-item-right">
                      <div className="rental-item-price">{rental.price} ₽</div>
                    </div>
                  </motion.div>
                ))
              ) : (
                <div className="empty-state">
                  <Bike size={64} color="#D1D5DB" />
                  <p>История аренд пуста</p>
                </div>
              )
            )}
          </div>

          <div className="slider-section">
            <ImageSlider />
          </div>
        </main>
      </div>
    </div>
  )
}

export default MyRentals


